<?php

namespace App\Models;

use CodeIgniter\Model;

class TicketsModel extends Model
{
    // Nombre de la tabla
    protected $table = 't_tickets';

    // Clave primaria
    protected $primaryKey = 'PK_Ticket';

    // Campos permitidos para inserción y actualización
    protected $allowedFields = [
        'FK_User',
        'Reason',
        'Category',
        'Priority',
        'CreatedAt',
        'AttendedAt',
        'Status',
        'Comments'
    ];

    // Activar timestamps automáticos para las columnas CreatedAt y UpdatedAt
    protected $useTimestamps = false; 
    protected $createdField = 'CreatedAt';
    protected $updatedField = 'AttendedAt';

    // Establecer validación para los campos
    protected $validationRules = [
        'FK_User'   => 'required|integer',
        'Reason'    => 'required|string',
        'Category'  => 'required|in_list[NuevaContraseña, Problema técnico, Consulta-general, Soporte, Otro]',
        'Priority'  => 'in_list[Baja, Media, Alta]',
        'Status'    => 'in_list[Pendiente, En proceso, Resuelto, Cancelado]'
    ];

    // Mensajes personalizados de error (opcional)
    protected $validationMessages = [
        'Category' => [
            'required' => 'La categoría es obligatoria',
            'in_list'  => 'La categoría debe ser una de las siguientes: Cambio de contraseña, Problema técnico, Consulta general, Otro'
        ],
        'Status' => [
            'in_list'  => 'El estado debe ser una de las siguientes opciones: Pendiente, En proceso, Resuelto, Cancelado'
        ]
    ];

    // Establecer el manejo de "soft deletes" para no eliminar tickets directamente
    protected $useSoftDeletes = true;
    protected $deletedField = 'DeletedAt';

    public function resolveTicket($ticketId)
    {
        // Usar el método update con el ID del ticket directamente
        return $this->update($ticketId, ['Status' => 'Resuelto', 'AttendedAt' => date('Y-m-d H:i:s')]);
    }
    

    // Métodos personalizados
    public function getTicketsByUser($userId)
    {
        return $this->where('FK_User', $userId)->findAll();
    }

    public function getPendingTickets()
    {
        return $this->where('Status', 'Pendiente')->findAll();
    }

    public function markAsInProcess($ticketId)
    {
        return $this->update($ticketId, ['Status' => 'En proceso', 'AttendedAt' => date('Y-m-d H:i:s')]);
    }


}
